use std::time::Duration;

use clap::{Arg, ArgAction, value_parser};
use daemonize::Daemonize;
use log::info;
use tokio::runtime::Builder;

use crate::calendar::{ScreenSaver, ScreenSaverCfg};

mod calendar;
mod fb;

fn main() {
    env_logger::init();
    let args = clap::Command::new("kindle-land")
        .arg(Arg::new("url")
            .long("url")
            .short('u')
            .value_parser(value_parser!(reqwest::Url))
            .action(ArgAction::Set)
            .required(true))
        .arg(Arg::new("daemon")
            .long("daemon")
            .short('d')
            .action(ArgAction::SetTrue))
        .arg(Arg::new("invert")
            .long("invert")
            .action(ArgAction::SetTrue))
        .arg(Arg::new("interval")
            .long("interval")
            .value_parser(value_parser!(u64))
            .default_value("300")
            .action(ArgAction::Set))
        .arg(Arg::new("mono")
            .long("mono")
            .action(ArgAction::SetTrue))
        .arg(Arg::new("rotate")
            .long("rotate")
            .action(ArgAction::SetTrue))
        .get_matches();
    if args.get_flag("daemon") {
        let daemonize = Daemonize::new()
            .pid_file("/tmp/kindle-land.pid") // Every method except `new` and `start`
            .chown_pid_file(true)      // is optional, see `Daemonize` documentation
            .working_directory("/tmp") // for default behaviour.
            .user("root")
            .group("root")
            .umask(0o777)    // Set umask, `0o027` by default.
            .privileged_action(|| "Executed before drop privileges");

        match daemonize.start() {
            Ok(_) => info!("success, daemonized"),
            Err(e) => panic!("daemonize error: {}", e)
        }
    }
    let mut flags = 0x00;
    if args.get_flag("invert") {
        flags |= 0x01;
    }
    if args.get_flag("mono") {
        flags |= 0x02;
    }


    let u: &reqwest::Url = args.get_one("url").unwrap();
    let d: &u64 = args.get_one("interval").unwrap();
    let cfg = ScreenSaverCfg {
        url: u.clone(),
        flags,
        duration: Duration::from_secs(d.clone()),
        rotate: args.get_flag("rotate"),
    };

    let rt = Builder::new_multi_thread().enable_all().build().unwrap();
    let _ = rt.block_on(async move {
        let saver = ScreenSaver::new(cfg).await.unwrap();
        saver.watch().await.unwrap();
    });
}

