use std::ffi::c_void;
use std::io;
use std::mem;
use std::mem::size_of;
use std::ptr;

use libc::{c_char, c_int, c_ulong, MAP_SHARED, O_RDWR, PROT_READ, PROT_WRITE};
use log::{debug, trace};

const FBIOGET_VSCREENINFO: c_ulong = 0x4600;
const FB_DEVICE: &str = "/dev/fb0";

#[repr(C)]
struct FBVarScreenInfo {
    x_res: u32,
    y_res: u32,
    x_res_virtual: u32,
    y_res_virtual: u32,
    x_offset: u32,
    y_offset: u32,
    bits_per_pixel: u32,
    grayscale: u32,
    reserved: [u32; 32],
}

unsafe fn get_fb_info(fd: c_int) -> io::Result<FBVarScreenInfo> {
    let mut vinfo = mem::zeroed();
    // const FBIOGET_VSCREENINFO: c_ulong = 0x4600;
    const FB_DEVICE: &str = "/dev/fb0";
    if libc::ioctl(fd, 0x4600, &mut vinfo) < 0 {
        Err(io::Error::last_os_error())
    } else {
        Ok(vinfo)
    }
}

#[repr(C)]
#[derive(Default)]
struct MxcfbRect {
    pub top: u32,
    pub left: u32,
    pub width: u32,
    pub height: u32,
}

#[repr(C)]
struct MxcfbAltBufferData {
    pub phys_addr: *const c_void,
    pub width: u32,
    pub height: u32,
    pub alt_update_region: MxcfbRect,
}

impl Default for MxcfbAltBufferData {
    fn default() -> Self {
        Self {
            phys_addr: ptr::null(),
            width: 0,
            height: 0,
            alt_update_region: Default::default(),
        }
    }
}

#[repr(C)]
struct MxcfbUpdateData {
    pub update_region: MxcfbRect,
    pub waveform_mode: u32,
    pub update_mode: u32,
    pub update_marker: u32,
    pub hist_bw_waveform_mode: u32,
    pub hist_gray_waveform_mode: u32,
    pub temp: i32,
    pub flags: u32,
    pub alt_buffer_data: MxcfbAltBufferData,
}

impl Default for MxcfbUpdateData {
    fn default() -> Self {
        Self {
            update_region: MxcfbRect {
                top: 0,
                left: 0,
                width: 1072,
                height: 1448,
            },
            waveform_mode: 0x02,
            update_mode: 0x01,
            update_marker: 0,
            hist_bw_waveform_mode: 0,
            hist_gray_waveform_mode: 0,
            temp: 0x1000,
            flags: 0,
            alt_buffer_data: MxcfbAltBufferData::default(),
        }
    }
}

impl FrameBufferScreen {
    pub fn new() -> io::Result<Self> {
        debug!("open framebuffer: {}", FB_DEVICE);
        let fb_fd =
            unsafe { libc::open(format!("{}", FB_DEVICE).as_ptr() as *const c_char, O_RDWR) };
        if fb_fd < 0 {
            return Err(io::Error::last_os_error());
        }

        let screen_info = match unsafe { get_fb_info(fb_fd) } {
            Ok(r) => r,
            Err(err) => {
                unsafe { libc::close(fb_fd) };
                return Err(err);
            }
        };
        debug!(
            "screen size: {}x{}, virtual size: {}x{} color: {} bits",
            screen_info.x_res,
            screen_info.y_res,
            screen_info.x_res_virtual,
            screen_info.y_res_virtual,
            screen_info.bits_per_pixel
        );
        let screen_size: usize = (screen_info.y_res_virtual
            * screen_info.x_res_virtual
            * (screen_info.bits_per_pixel / 8)) as usize;

        let framebuffer = unsafe {
            libc::mmap(
                ptr::null_mut(),
                screen_size,
                PROT_READ | PROT_WRITE,
                MAP_SHARED,
                fb_fd,
                0,
            ) as *mut u8
        };
        if framebuffer == ptr::null_mut() {
            unsafe { libc::close(fb_fd) };
            return Err(io::Error::last_os_error());
        }
        debug!("mmap buffer: {} bytes", screen_size);

        Ok(Self {
            x_real: screen_info.x_res as usize,
            y_real: screen_info.y_res as usize,
            fd: fb_fd,
            x: screen_info.x_res_virtual,
            y: screen_info.y_res_virtual,
            color_bits: screen_info.bits_per_pixel as usize,
            screen_size,
            framebuffer_map: framebuffer,
        })
    }
}

pub struct FrameBufferScreen {
    fd: c_int,
    pub x_real: usize,
    pub y_real: usize,
    pub x: u32,
    pub y: u32,
    pub color_bits: usize,
    pub screen_size: usize,
    framebuffer_map: *const u8,
}
unsafe impl Send for FrameBufferScreen {}

const IO_R: libc::c_int = 1 << 31;
const IO_W: libc::c_int = 1 << 30;

#[inline]
unsafe fn kindle_refresh(fd: libc::c_int, data: &MxcfbUpdateData) -> c_int {
    let data = data as *const _ as *const libc::c_void;
    let size: libc::c_int = (size_of::<MxcfbUpdateData>() as c_int) << 16;
    let magic: libc::c_int = (b'F' as c_int) << 8;
    let cmd: libc::c_int = 0x2e;
    let q: libc::c_int = IO_W | size | magic | cmd;
    libc::ioctl(fd, q.try_into().unwrap(), data)
}

impl FrameBufferScreen {
    pub fn refresh_full(&self) -> io::Result<()> {
        self.refresh_full_with_flag(0)
    }
    pub fn refresh_full_with_flag(&self, flags: u32) -> io::Result<()> {
        trace!("framebuffer refresh");
        let d = MxcfbUpdateData {
            flags,
            ..MxcfbUpdateData::default()
        };
        if unsafe { kindle_refresh(self.fd, &d) } != 0 {
            Err(io::Error::last_os_error())
        } else {
            Ok(())
        }
    }
    pub fn update_u32(&self, x: u32, y: u32, c: u32) {
        if x >= self.x || y >= self.y {
            return;
        }
        if self.color_bits == 8 {
            let argb = c;
            let a = (argb >> 24) & 0xff;
            let r = (argb >> 16) & 0xff;
            let g = (argb >> 8) & 0xff;
            let b = argb & 0xff;

            let gray = (r as f32 * 0.30 + g as f32 * 0.59 + b as f32 * 0.11) as u8;
            unsafe {
                (self.framebuffer_map as *mut u8)
                    .add((self.x * y + x) as usize)
                    .write(gray);
            }
        }
    }

    pub fn update(&self, x: u32, y: u32, c: u8) {
        if x >= self.x || y >= self.y {
            return;
        }
        if self.color_bits == 8 {
            unsafe {
                (self.framebuffer_map as *mut u8)
                    .add((self.x * y + x) as usize)
                    .write(c);
            }
        }
    }
}

impl Drop for FrameBufferScreen {
    fn drop(&mut self) {
        if self.framebuffer_map != ptr::null() {
            unsafe { libc::munmap(self.framebuffer_map as *mut c_void, self.screen_size) };
        }
        if self.fd != 0 {
            unsafe { libc::close(self.fd) };
        }
    }
}
