use std::process::Stdio;
use std::sync::Arc;
use std::sync::atomic::{AtomicBool, Ordering};
use std::time::Duration;

use anyhow::anyhow;
use image::{GrayImage, ImageFormat, imageops};
use log::{debug, info, warn};
use tokio::io::{AsyncBufReadExt, AsyncReadExt, BufReader};
use tokio::process::Command;
use tokio::sync::Mutex;

use crate::fb;

enum Action {
    Update,
    Render,
    Quit,
}

pub(crate) struct ScreenSaver {
    lock: Mutex<()>,
    busy: AtomicBool,
    duration_update: Duration,
    bus: tokio::sync::mpsc::Sender<Action>,
}

async fn wait_network() -> anyhow::Result<bool> {
    let child = Command::new("lipc-get-prop")
        .arg("com.lab126.wifid")
        .arg("cmState").stdout(Stdio::piped()).spawn()?;
    let out = child.wait_with_output().await?;
    if String::from_utf8_lossy(out.stdout.as_slice()).trim() == "CONNECTED" {
        return Ok(true);
    }
    Command::new("lipc-set-prop")
        .arg("com.lab126.cmd")
        .arg("wirelessEnable")
        .arg("1")
        .spawn()?
        .wait()
        .await?;
    debug!("waiting wifi connect");
    Command::new("lipc-wait-event")
        .arg("-s").arg("30").arg("com.lab126.wifid")
        .arg("cmConnected").spawn()?.wait().await?;
    let out = Command::new("lipc-get-prop")
        .arg("com.lab126.wifid")
        .arg("cmState").stdout(Stdio::piped())
        .spawn()?
        .wait_with_output().await?
        .stdout;
    let out = String::from_utf8_lossy(out.as_slice());
    return Ok(out.trim() == "CONNECTED");
}

async fn daemon(cover: CoverFetcher, flags: u32) -> anyhow::Result<tokio::sync::mpsc::Sender<Action>> {
    let mut picture = cover.fetch().await?;
    let fb = fb::FrameBufferScreen::new().unwrap();
    let (tx, mut rx) = tokio::sync::mpsc::channel::<Action>(1);
    tokio::spawn(async move {
        while let Some(action) = rx.recv().await {
            match action {
                Action::Update => {
                    debug!("action:update");
                    let connecting = wait_network().await.unwrap();
                    if !connecting {
                        info!("not network, skip update");
                        continue;
                    }
                    tokio::time::sleep(Duration::from_secs(5)).await;
                    match cover.fetch().await {
                        Ok(c) => {
                            picture = c;
                        }
                        Err(err) => {
                            warn!("picture download error: {}",err);
                        }
                    }
                }
                Action::Render => {
                    debug!("action:render");
                    picture.enumerate_rows()
                        .flat_map(|(_, row)| row)
                        .for_each(|(x, y, p)| {
                            let [gray] = p.0;
                            fb.update(x, y, gray);
                        });
                    fb.refresh_full_with_flag(flags).unwrap();
                }
                Action::Quit => { return; }
            }
        };
        unreachable!()
    });
    Ok(tx)
}

struct CoverFetcher {
    url: reqwest::Url,
    rotate: bool,
    w: u32,
    h: u32,
}

impl CoverFetcher {
    async fn fetch(&self) -> anyhow::Result<GrayImage> {
        let response = reqwest::get(self.url.clone()).await?;
        if !response.status().is_success() {
            return Err(anyhow!("Request failed with status code: {}", response.status()));
        }
        let mut img = match response.headers().get("Content-Type").and_then(|v| v.to_str().ok()) {
            Some("image/png") => {
                image::load_from_memory_with_format(response.bytes().await?.as_ref(), ImageFormat::Png)
            }
            Some("image/jpeg") => {
                image::load_from_memory_with_format(response.bytes().await?.as_ref(), ImageFormat::Jpeg)
            }
            Some("image/gif") => {
                image::load_from_memory_with_format(response.bytes().await?.as_ref(), ImageFormat::Gif)
            }
            Some(f) => {
                return Err(anyhow!("unsupported format: {}",f));
            }
            None => {
                return Err(anyhow!("Content-Type not found"));
            }
        }?;
        info!("image size: {}x{}",img.width(),img.height());
        if self.rotate {
            img = img.rotate90();
        }
        Ok(img.resize_exact(self.w, self.h, imageops::FilterType::Nearest).into_luma8())
    }
}

pub struct ScreenSaverCfg {
    pub url: reqwest::Url,
    pub flags: u32,
    pub duration: Duration,
    pub rotate: bool,
}

impl Default for ScreenSaverCfg {
    fn default() -> Self {
        Self {
            url: reqwest::Url::parse("http://localhost:8080/kindle.png").unwrap(),
            flags: 0,
            duration: Duration::from_secs(3600),
            rotate: false,
        }
    }
}

impl ScreenSaver {
    pub async fn new(cfg: ScreenSaverCfg) -> anyhow::Result<Arc<Self>> {
        let cover = CoverFetcher { url: cfg.url, rotate: cfg.rotate, w: 1074, h: 1448 };
        let send = daemon(cover, cfg.flags).await?;
        Ok(Arc::new(Self {
            lock: Mutex::new(()),
            busy: AtomicBool::new(true),
            duration_update: cfg.duration,
            bus: send,
        }))
    }
    async fn render_delay(self: &Arc<Self>, update: bool) -> anyhow::Result<()> {
        tokio::time::sleep(Duration::from_secs(3)).await;
        if self.busy.load(Ordering::Relaxed) {
            return Ok(());
        }
        if update {
            self.bus.send(Action::Update).await?;
        }
        self.bus.send(Action::Render).await?;
        return Ok(());
    }
    pub async fn watch(self: &Arc<Self>) -> anyhow::Result<()> {
        let _ = self.lock.lock().await;
        debug!("watching...");
        let mut cmd = Command::new("/usr/bin/lipc-wait-event")
            .arg("-m")
            .arg("com.lab126.powerd")
            .arg("*")
            .stdout(Stdio::piped())
            .spawn()?;
        let stdout = cmd.stdout.take().expect("Failed to open stdout");

        let mut lines = BufReader::new(stdout).lines();
        while let Some(line) = lines.next_line().await? {
            let cmd = match line.trim().split_once(" ") {
                Some((t, _)) => { t }
                _ => { line.trim() }
            };
            debug!("power event: {}",cmd);
            match cmd {
                "battLevelChanged" => {}
                "goingToScreenSaver" => {
                    self.busy.store(false, Ordering::Relaxed);
                    let cp = self.clone();
                    tokio::spawn(async move {
                        cp.render_delay(false).await.unwrap();
                    });
                }
                "wakeupFromSuspend" => {
                    let cp = self.clone();
                    tokio::spawn(async move {
                        cp.render_delay(true).await.unwrap();
                    });
                }
                "outOfScreenSaver" => {
                    self.busy.store(true, Ordering::Relaxed);
                }
                "exitingScreenSaver" => {
                    wifi(Wireless::Enable).await.unwrap();
                }
                "readyToSuspend" => {
                    wifi(Wireless::Disable).await.unwrap();
                    Command::new("lipc-set-prop")
                        .arg("-i").arg("com.lab126.powerd").arg("rtcWakeup")
                        .arg(format!("{}", self.duration_update.as_secs()))
                        .spawn().unwrap().wait().await.unwrap();
                }
                "suspending" => {}
                "resuming" => {}
                "t1TimerReset" => {}
                other => {
                    warn!("unknown power event: {}",other);
                }
            };
        }
        unreachable!()
    }
}

impl Drop for ScreenSaver {
    fn drop(&mut self) {
        self.bus.blocking_send(Action::Quit).unwrap();
    }
}

enum Wireless {
    Enable,
    Disable,
}

async fn wifi(action: Wireless) -> anyhow::Result<()> {
    let v = match action {
        Wireless::Enable => { "1" }
        Wireless::Disable => { "0" }
    };
    let out = Command::new("lipc-set-prop")
        .arg("com.lab126.cmd")
        .arg("wirelessEnable")
        .arg(v)
        .stdout(Stdio::piped())
        .stderr(Stdio::piped())
        .spawn()?
        .wait_with_output()
        .await?;

    if !out.status.success() {
        let out = String::from_utf8_lossy(out.stderr.as_slice());
        return Err(anyhow!("enable wifi error:{out}"));
    }
    return Ok(());
}